// Particle Animation
const canvas = document.getElementById('particles-canvas');
const ctx = canvas.getContext('2d');
canvas.width = window.innerWidth;
canvas.height = window.innerHeight;

// Configuration
const particleConfig = {
    particleCount: 100,
    particleColor: '#4ca5ff',
    secondaryColor: '#b673f8',
    lineColor: 'rgba(76, 165, 255, 0.15)',
    particleRadius: 2,
    lineWidth: 1,
    lineDistance: 150,
    particleSpeed: 0.5,
    interactivity: true,
    interactiveRadius: 100
};

// Responsive handling
window.addEventListener('resize', () => {
    canvas.width = window.innerWidth;
    canvas.height = window.innerHeight;
    initParticles();
});

// Mouse position tracking
let mouse = {
    x: null,
    y: null,
    radius: particleConfig.interactiveRadius
};

window.addEventListener('mousemove', (event) => {
    mouse.x = event.x;
    mouse.y = event.y;
});

window.addEventListener('mouseout', () => {
    mouse.x = undefined;
    mouse.y = undefined;
});

// Particle class
class Particle {
    constructor() {
        this.x = Math.random() * canvas.width;
        this.y = Math.random() * canvas.height;
        this.size = Math.random() * particleConfig.particleRadius + 1;
        this.speedX = (Math.random() - 0.5) * particleConfig.particleSpeed;
        this.speedY = (Math.random() - 0.5) * particleConfig.particleSpeed;
        this.color = Math.random() > 0.5 ? particleConfig.particleColor : particleConfig.secondaryColor;
    }

    update() {
        // Movement
        this.x += this.speedX;
        this.y += this.speedY;

        // Boundary checking
        if (this.x > canvas.width || this.x < 0) {
            this.speedX = -this.speedX;
        }
        if (this.y > canvas.height || this.y < 0) {
            this.speedY = -this.speedY;
        }

        // Mouse interactivity
        if (particleConfig.interactivity && mouse.x != undefined && mouse.y != undefined) {
            const dx = mouse.x - this.x;
            const dy = mouse.y - this.y;
            const distance = Math.sqrt(dx * dx + dy * dy);

            if (distance < mouse.radius) {
                const forceX = dx / 10;
                const forceY = dy / 10;
                this.x -= forceX;
                this.y -= forceY;
            }
        }
    }

    draw() {
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
        ctx.fillStyle = this.color;
        ctx.fill();
    }
}

let particles = [];

function initParticles() {
    particles = [];
    for (let i = 0; i < particleConfig.particleCount; i++) {
        particles.push(new Particle());
    }
}

function connectParticles() {
    for (let i = 0; i < particles.length; i++) {
        for (let j = i; j < particles.length; j++) {
            const dx = particles[i].x - particles[j].x;
            const dy = particles[i].y - particles[j].y;
            const distance = Math.sqrt(dx * dx + dy * dy);

            if (distance < particleConfig.lineDistance) {
                // Create gradient lines that fade with distance
                const opacity = 1 - (distance / particleConfig.lineDistance);
                ctx.beginPath();
                ctx.strokeStyle = particleConfig.lineColor;
                ctx.lineWidth = particleConfig.lineWidth;
                ctx.moveTo(particles[i].x, particles[i].y);
                ctx.lineTo(particles[j].x, particles[j].y);
                ctx.stroke();
            }
        }
    }
}

function animate() {
    requestAnimationFrame(animate);
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    for (let i = 0; i < particles.length; i++) {
        particles[i].update();
        particles[i].draw();
    }

    connectParticles();
}

initParticles();
animate();